"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenIdAuthentication = void 0;
var fs = _interopRequireWildcard(require("fs"));
var _wreck = _interopRequireDefault(require("@hapi/wreck"));
var _proxyAgent = require("proxy-agent");
var _security_cookie = require("../../../session/security_cookie");
var _routes = require("./routes");
var _authentication_type = require("../authentication_type");
var _helper = require("./helper");
var _object_properties_defined = require("../../../utils/object_properties_defined");
var _common = require("../../../../common");
var _cookie_splitter = require("../../../session/cookie_splitter");
var _http = require("../../../../../../src/core/server/http");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class OpenIdAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, core, logger) {
    var _this$config$openid, _this$config$openid2;
    super(config, sessionStorageFactory, router, esClient, core, logger);
    _defineProperty(this, "type", _common.AuthType.OPEN_ID);
    _defineProperty(this, "openIdAuthConfig", void 0);
    _defineProperty(this, "authHeaderName", void 0);
    _defineProperty(this, "openIdConnectUrl", void 0);
    _defineProperty(this, "wreckClient", void 0);
    _defineProperty(this, "wreckHttpsOption", {});
    _defineProperty(this, "redirectOIDCCapture", (request, toolkit) => {
      const nextUrl = this.generateNextUrl(request);
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return toolkit.redirected({
        location: `${this.coreSetup.http.basePath.serverBasePath}/auth/openid/captureUrlFragment?nextUrl=${nextUrl}`,
        'set-cookie': clearOldVersionCookie
      });
    });
    this.wreckClient = this.createWreckClient();
    this.openIdAuthConfig = {};
    this.authHeaderName = ((_this$config$openid = this.config.openid) === null || _this$config$openid === void 0 ? void 0 : _this$config$openid.header) || '';
    this.openIdAuthConfig.authHeaderName = this.authHeaderName;
    this.openIdConnectUrl = ((_this$config$openid2 = this.config.openid) === null || _this$config$openid2 === void 0 ? void 0 : _this$config$openid2.connect_url) || '';
    let scope = this.config.openid.scope;
    if (scope.indexOf('openid') < 0) {
      scope = `openid ${scope}`;
    }
    this.openIdAuthConfig.scope = scope;
  }
  async init() {
    try {
      const response = await this.wreckClient.get(this.openIdConnectUrl);
      const payload = JSON.parse(response.payload);
      this.openIdAuthConfig.authorizationEndpoint = payload.authorization_endpoint;
      this.openIdAuthConfig.tokenEndpoint = payload.token_endpoint;
      this.openIdAuthConfig.endSessionEndpoint = payload.end_session_endpoint || undefined;
      this.createExtraStorage();
      const routes = new _routes.OpenIdAuthRoutes(this.router, this.config, this.sessionStorageFactory, this.openIdAuthConfig, this.securityClient, this.coreSetup, this.wreckClient);
      routes.setupRoutes();
    } catch (error) {
      this.logger.error(error); // TODO: log more info
      throw new Error('Failed when trying to obtain the endpoints from your IdP');
    }
  }
  generateNextUrl(request) {
    const path = (0, _http.getRedirectUrl)({
      request,
      basePath: this.coreSetup.http.basePath.serverBasePath,
      nextUrl: request.url.pathname || '/app/opensearch-dashboards'
    });
    return escape(path);
  }
  createWreckClient() {
    var _this$config$openid3, _this$config$openid4, _this$config$openid5, _this$config$openid6, _this$config$openid7, _this$config$openid9;
    if ((_this$config$openid3 = this.config.openid) !== null && _this$config$openid3 !== void 0 && _this$config$openid3.root_ca) {
      this.wreckHttpsOption.ca = [fs.readFileSync(this.config.openid.root_ca)];
      this.logger.debug(`Using CA Cert: ${this.config.openid.root_ca}`);
    }
    if ((_this$config$openid4 = this.config.openid) !== null && _this$config$openid4 !== void 0 && _this$config$openid4.pfx) {
      // Use PFX or PKCS12 if provided
      this.logger.debug(`Using PFX or PKCS12: ${this.config.openid.pfx}`);
      this.wreckHttpsOption.pfx = [fs.readFileSync(this.config.openid.pfx)];
    } else if ((_this$config$openid5 = this.config.openid) !== null && _this$config$openid5 !== void 0 && _this$config$openid5.certificate && (_this$config$openid6 = this.config.openid) !== null && _this$config$openid6 !== void 0 && _this$config$openid6.private_key) {
      // Use 'certificate' and 'private_key' if provided
      this.logger.debug(`Using Certificate: ${this.config.openid.certificate}`);
      this.logger.debug(`Using Private Key: ${this.config.openid.private_key}`);
      this.wreckHttpsOption.cert = [fs.readFileSync(this.config.openid.certificate)];
      this.wreckHttpsOption.key = [fs.readFileSync(this.config.openid.private_key)];
    } else {
      this.logger.debug(`Client certificates not provided. Mutual TLS will not be used to obtain endpoints.`);
    }
    // Check if passphrase is provided, use it for 'pfx' and 'key'
    if (((_this$config$openid7 = this.config.openid) === null || _this$config$openid7 === void 0 ? void 0 : _this$config$openid7.passphrase) !== '') {
      var _this$config$openid8;
      this.logger.debug(`Passphrase not provided for private key and/or pfx.`);
      this.wreckHttpsOption.passphrase = (_this$config$openid8 = this.config.openid) === null || _this$config$openid8 === void 0 ? void 0 : _this$config$openid8.passphrase;
    }
    if (((_this$config$openid9 = this.config.openid) === null || _this$config$openid9 === void 0 ? void 0 : _this$config$openid9.verify_hostnames) === false) {
      this.logger.debug(`openId auth 'verify_hostnames' option is off.`);
      this.wreckHttpsOption.checkServerIdentity = (host, cert) => {
        return undefined;
      };
    }
    this.logger.info((0, _object_properties_defined.getObjectProperties)(this.wreckHttpsOption, 'WreckHttpsOptions'));

    // Use proxy agent to allow usage of e.g. http_proxy environment variable
    const httpAgent = new _proxyAgent.ProxyAgent();
    const httpsAllowUnauthorizedAgent = new _proxyAgent.ProxyAgent({
      rejectUnauthorized: false
    });
    let httpsAgent = new _proxyAgent.ProxyAgent();
    if (Object.keys(this.wreckHttpsOption).length > 0) {
      httpsAgent = new _proxyAgent.ProxyAgent(this.wreckHttpsOption);
    }
    return _wreck.default.defaults({
      agents: {
        http: httpAgent,
        https: httpsAgent,
        httpsAllowUnauthorized: httpsAllowUnauthorizedAgent
      }
    });
  }
  getWreckHttpsOptions() {
    return this.wreckHttpsOption;
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const extraCookiePrefix = this.config.openid.extra_storage.cookie_prefix;
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= this.config.openid.extra_storage.additional_cookies; i++) {
      hapiServer.states.add(extraCookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions() {
    // If we're here, we will always have the openid configuration
    return {
      cookiePrefix: this.config.openid.extra_storage.cookie_prefix,
      additionalCookies: this.config.openid.extra_storage.additional_cookies,
      logger: this.logger
    };
  }
  requestIncludesAuthInfo(request) {
    return request.headers.authorization ? true : false;
  }
  async getAdditionalAuthHeader(request) {
    return {};
  }
  getCookie(request, authInfo) {
    (0, _cookie_splitter.setExtraAuthStorage)(request, request.headers.authorization, this.getExtraAuthStorageOptions());
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: this.type,
      expiryTime: Date.now() + this.config.session.ttl
    };
  }
  getKeepAliveExpiry(cookie, request) {
    return Date.now() + this.config.session.ttl;
  }

  // TODO: Add token expiration check here
  async isValidCookie(cookie, request) {
    var _cookie$credentials;
    if (cookie.authType !== this.type || !cookie.username || !cookie.expiryTime || !((_cookie$credentials = cookie.credentials) !== null && _cookie$credentials !== void 0 && _cookie$credentials.authHeaderValue) && !this.getExtraAuthStorageValue(request, cookie)) {
      return false;
    }
    if (cookie.credentials.expiryTime > Date.now()) {
      return true;
    }

    // need to renew id token
    if (cookie.credentials.refresh_token) {
      try {
        var _this$config$openid10, _this$config$openid11;
        const query = {
          grant_type: 'refresh_token',
          client_id: (_this$config$openid10 = this.config.openid) === null || _this$config$openid10 === void 0 ? void 0 : _this$config$openid10.client_id,
          client_secret: (_this$config$openid11 = this.config.openid) === null || _this$config$openid11 === void 0 ? void 0 : _this$config$openid11.client_secret,
          refresh_token: cookie.credentials.refresh_token
        };
        const refreshTokenResponse = await (0, _helper.callTokenEndpoint)(this.openIdAuthConfig.tokenEndpoint, query, this.wreckClient);

        // if no id_token from refresh token call, maybe the Idp doesn't allow refresh id_token
        if (refreshTokenResponse.idToken) {
          cookie.credentials = {
            authHeaderValueExtra: true,
            refresh_token: refreshTokenResponse.refreshToken,
            expiryTime: (0, _helper.getExpirationDate)(refreshTokenResponse)
          };
          (0, _cookie_splitter.setExtraAuthStorage)(request, `Bearer ${refreshTokenResponse.idToken}`, this.getExtraAuthStorageOptions());
          return true;
        } else {
          return false;
        }
      } catch (error) {
        this.logger.error(error);
        return false;
      }
    } else {
      // no refresh token, and current token is expired
      return false;
    }
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      return this.redirectOIDCCapture(request, toolkit);
    } else {
      return response.unauthorized();
    }
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;
    let extraValue = '';
    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions());
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;
    const header = {};
    if (cookie.credentials.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        header.authorization = extraAuthStorageValue;
        return header;
      } catch (error) {
        this.logger.error(error);
        // TODO Re-throw?
        // throw error;
      }
    }

    const authHeaderValue = (_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.authHeaderValue;
    if (authHeaderValue) {
      header.authorization = authHeaderValue;
    }
    return header;
  }
}
exports.OpenIdAuthentication = OpenIdAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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