"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDslRoute = registerDslRoute;
var _configSchema = require("@osd/config-schema");
var _shared = require("../../common/constants/shared");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerDslRoute({
  router
}, dataSourceEnabled) {
  router.post({
    path: `${_shared.DSL_BASE}${_shared.DSL_SEARCH}`,
    validate: {
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    const {
      index,
      size,
      ...rest
    } = request.body;
    const params = {
      index,
      size,
      body: rest
    };
    try {
      const resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_shared.DSL_BASE}${_shared.DSL_CAT}`,
    validate: {
      query: _configSchema.schema.object({
        format: _configSchema.schema.string(),
        index: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const resp = await context.core.opensearch.legacy.client.callAsCurrentUser('cat.indices', request.query);
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_shared.DSL_BASE}${_shared.DSL_MAPPING}`,
    validate: {
      query: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    try {
      const resp = await context.core.opensearch.legacy.client.callAsCurrentUser('indices.getMapping', {
        index: request.query.index
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_shared.DSL_BASE}${_shared.DSL_SETTINGS}`,
    validate: {
      query: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    try {
      const resp = await context.core.opensearch.legacy.client.callAsCurrentUser('indices.getSettings', {
        index: request.query.index
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_shared.DSL_BASE}${_shared.DSL_CAT}/dataSourceMDSId={dataSourceMDSId?}`,
    validate: {
      query: _configSchema.schema.object({
        format: _configSchema.schema.string(),
        index: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      params: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const dataSourceMDSId = request.params.dataSourceMDSId;
    try {
      let resp;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = await context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('cat.indices', request.query);
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('cat.indices', request.query);
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_shared.DSL_BASE}${_shared.DSL_MAPPING}/dataSourceMDSId={dataSourceMDSId?}`,
    validate: {
      query: _configSchema.schema.any(),
      params: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const dataSourceMDSId = request.params.dataSourceMDSId;
    try {
      let resp;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = await context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('indices.getMapping', {
          index: request.query.index
        });
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('indices.getMapping', {
          index: request.query.index
        });
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_shared.DSL_BASE}${_shared.DSL_SETTINGS}/dataSourceMDSId={dataSourceMDSId?}`,
    validate: {
      query: _configSchema.schema.any(),
      params: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const dataSourceMDSId = request.params.dataSourceMDSId;
    try {
      let resp;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = await context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('indices.getSettings', {
          index: request.query.index
        });
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('indices.getSettings', {
          index: request.query.index
        });
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_shared.DSL_BASE}/integrations/refresh`,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const dataSourceMDSId = request.query.dataSourceMDSId;
    try {
      let resp;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = await context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('indices.refresh');
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('indices.refresh');
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_shared.DSL_BASE}/integrations/mapping`,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const dataSourceMDSId = request.query.dataSourceMDSId;
    try {
      let resp;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = await context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('indices.getMapping');
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('indices.getMapping');
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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