/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.consumer.internals;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import org.apache.kafka.clients.consumer.internals.ConsumerMetadata;
import org.apache.kafka.clients.consumer.internals.MemberState;
import org.apache.kafka.clients.consumer.internals.MemberStateListener;
import org.apache.kafka.clients.consumer.internals.NetworkClientDelegate;
import org.apache.kafka.clients.consumer.internals.RequestManager;
import org.apache.kafka.clients.consumer.internals.SubscriptionState;
import org.apache.kafka.clients.consumer.internals.Utils;
import org.apache.kafka.clients.consumer.internals.metrics.ShareRebalanceMetricsManager;
import org.apache.kafka.common.TopicIdPartition;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.message.ShareGroupHeartbeatResponseData;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.telemetry.internals.ClientTelemetryReporter;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;

public class ShareMembershipManager
implements RequestManager {
    private final Logger log;
    static final Utils.TopicPartitionComparator TOPIC_PARTITION_COMPARATOR = new Utils.TopicPartitionComparator();
    static final Utils.TopicIdPartitionComparator TOPIC_ID_PARTITION_COMPARATOR = new Utils.TopicIdPartitionComparator();
    private final String groupId;
    private String memberId = "";
    private int memberEpoch = 0;
    private final String rackId;
    private MemberState state;
    private Map<Uuid, SortedSet<Integer>> currentAssignment;
    private final SubscriptionState subscriptions;
    private final ConsumerMetadata metadata;
    private final Map<Uuid, String> assignedTopicNamesCache;
    private final Map<Uuid, SortedSet<Integer>> currentTargetAssignment;
    private boolean reconciliationInProgress;
    private boolean rejoinedWhileReconciliationInProgress;
    private Optional<CompletableFuture<Void>> leaveGroupInProgress = Optional.empty();
    private final List<MemberStateListener> stateUpdatesListeners;
    private final Optional<ClientTelemetryReporter> clientTelemetryReporter;
    private final ShareRebalanceMetricsManager metricsManager;
    private final Time time;
    private boolean isPollTimerExpired;

    public ShareMembershipManager(LogContext logContext, String groupId, String rackId, SubscriptionState subscriptions, ConsumerMetadata metadata, Optional<ClientTelemetryReporter> clientTelemetryReporter, Time time, Metrics metrics) {
        this(logContext, groupId, rackId, subscriptions, metadata, clientTelemetryReporter, time, new ShareRebalanceMetricsManager(metrics));
    }

    ShareMembershipManager(LogContext logContext, String groupId, String rackId, SubscriptionState subscriptions, ConsumerMetadata metadata, Optional<ClientTelemetryReporter> clientTelemetryReporter, Time time, ShareRebalanceMetricsManager metricsManager) {
        this.log = logContext.logger(ShareMembershipManager.class);
        this.groupId = groupId;
        this.rackId = rackId;
        this.state = MemberState.UNSUBSCRIBED;
        this.subscriptions = subscriptions;
        this.metadata = metadata;
        this.assignedTopicNamesCache = new HashMap<Uuid, String>();
        this.currentTargetAssignment = new HashMap<Uuid, SortedSet<Integer>>();
        this.currentAssignment = new HashMap<Uuid, SortedSet<Integer>>();
        this.stateUpdatesListeners = new ArrayList<MemberStateListener>();
        this.clientTelemetryReporter = clientTelemetryReporter;
        this.time = time;
        this.metricsManager = metricsManager;
    }

    private void transitionTo(MemberState nextState) {
        if (!this.state.equals((Object)nextState) && !nextState.getPreviousValidStates().contains((Object)this.state)) {
            throw new IllegalStateException(String.format("Invalid state transition from %s to %s", new Object[]{this.state, nextState}));
        }
        if (ShareMembershipManager.isCompletingRebalance(this.state, nextState)) {
            this.metricsManager.recordRebalanceEnded(this.time.milliseconds());
        }
        if (ShareMembershipManager.isStartingRebalance(this.state, nextState)) {
            this.metricsManager.recordRebalanceStarted(this.time.milliseconds());
        }
        this.log.trace("Member {} with epoch {} transitioned from {} to {}.", new Object[]{this.memberId, this.memberEpoch, this.state, nextState});
        this.state = nextState;
    }

    private static boolean isCompletingRebalance(MemberState currentState, MemberState nextState) {
        return currentState == MemberState.RECONCILING && (nextState == MemberState.STABLE || nextState == MemberState.ACKNOWLEDGING);
    }

    private static boolean isStartingRebalance(MemberState currentState, MemberState nextState) {
        return currentState != MemberState.RECONCILING && nextState == MemberState.RECONCILING;
    }

    public String groupId() {
        return this.groupId;
    }

    public String memberId() {
        return this.memberId;
    }

    public int memberEpoch() {
        return this.memberEpoch;
    }

    public String rackId() {
        return this.rackId;
    }

    public void onHeartbeatSuccess(ShareGroupHeartbeatResponseData response) {
        if (response.errorCode() != Errors.NONE.code()) {
            String errorMessage = String.format("Unexpected error in Heartbeat response. Expected no error, but received: %s", new Object[]{Errors.forCode(response.errorCode())});
            throw new IllegalArgumentException(errorMessage);
        }
        MemberState state = this.state();
        if (state == MemberState.LEAVING) {
            this.log.debug("Ignoring heartbeat response received from broker. Member {} with epoch {} is already leaving the group.", (Object)this.memberId, (Object)this.memberEpoch);
            return;
        }
        if (this.isNotInGroup()) {
            this.log.debug("Ignoring heartbeat response received from broker. Member {} is in {} state so it's not a member of the group. ", (Object)this.memberId, (Object)state);
            return;
        }
        if (response.memberId() != null && !response.memberId().equals(this.memberId)) {
            this.clientTelemetryReporter.ifPresent(reporter -> reporter.updateMetricsLabels(Collections.singletonMap("group_member_id", response.memberId())));
        }
        this.memberId = response.memberId();
        this.updateMemberEpoch(response.memberEpoch());
        ShareGroupHeartbeatResponseData.Assignment assignment = response.assignment();
        if (assignment != null) {
            if (!state.canHandleNewAssignment()) {
                this.log.debug("Ignoring new assignment {} received from server because member is in {} state.", (Object)assignment, (Object)state);
                return;
            }
            this.processAssignmentReceived(assignment);
        }
    }

    private boolean isNotInGroup() {
        return this.state == MemberState.UNSUBSCRIBED || this.state == MemberState.FENCED || this.state == MemberState.FATAL || this.state == MemberState.STALE;
    }

    private void processAssignmentReceived(ShareGroupHeartbeatResponseData.Assignment assignment) {
        this.replaceTargetAssignmentWithNewAssignment(assignment);
        if (!this.targetAssignmentReconciled()) {
            this.transitionTo(MemberState.RECONCILING);
        } else {
            this.log.debug("Target assignment {} received from the broker is equals to the member current assignment {}. Nothing to reconcile.", this.currentTargetAssignment, this.currentAssignment);
            if (this.state == MemberState.RECONCILING || this.state == MemberState.JOINING) {
                this.transitionTo(MemberState.STABLE);
            }
        }
    }

    private void replaceTargetAssignmentWithNewAssignment(ShareGroupHeartbeatResponseData.Assignment assignment) {
        this.currentTargetAssignment.clear();
        assignment.topicPartitions().forEach(topicPartitions -> this.currentTargetAssignment.put(topicPartitions.topicId(), new TreeSet<Integer>(topicPartitions.partitions())));
    }

    public void transitionToFenced() {
        if (this.state == MemberState.PREPARE_LEAVING) {
            this.log.debug("Member {} with epoch {} got fenced but it is already preparing to leave the group, so it will stop sending heartbeat and won't attempt to rejoin.", (Object)this.memberId, (Object)this.memberEpoch);
            this.transitionTo(MemberState.UNSUBSCRIBED);
            return;
        }
        if (this.state == MemberState.LEAVING) {
            this.log.debug("Member {} with epoch {} got fenced but it is already leaving the group with state {}, so it won't attempt to rejoin.", new Object[]{this.memberId, this.memberEpoch, this.state});
            return;
        }
        if (this.state == MemberState.UNSUBSCRIBED) {
            this.log.debug("Member {} with epoch {} got fenced but it already left the group, so it won't attempt to rejoin.", (Object)this.memberId, (Object)this.memberEpoch);
            return;
        }
        this.transitionTo(MemberState.FENCED);
        this.resetEpoch();
        this.log.debug("Member {} with epoch {} transitioned to {} state. It will release its assignment and rejoin the group.", new Object[]{this.memberId, this.memberEpoch, MemberState.FENCED});
        this.clearSubscription();
        this.transitionToJoining();
    }

    public void transitionToFatal() {
        MemberState previousState = this.state;
        this.transitionTo(MemberState.FATAL);
        if (this.memberId.isEmpty()) {
            this.log.error("Member {} with epoch {} transitioned to {} state", new Object[]{this.memberId, this.memberEpoch, MemberState.FATAL});
        } else {
            this.log.error("Non-member transitioned to {} state", (Object)MemberState.FATAL);
        }
        this.notifyEpochChange(Optional.empty(), Optional.empty());
        if (previousState == MemberState.UNSUBSCRIBED) {
            this.log.debug("Member {} with epoch {} got fatal error from the broker but it already left the group.", (Object)this.memberId, (Object)this.memberEpoch);
            return;
        }
        this.clearSubscription();
    }

    public void onSubscriptionUpdated() {
        if (this.state == MemberState.UNSUBSCRIBED) {
            this.transitionToJoining();
        }
    }

    private void clearSubscription() {
        if (this.subscriptions.hasAutoAssignedPartitions()) {
            this.subscriptions.assignFromSubscribed(Collections.emptySet());
        }
        this.updateAssignment(Collections.emptySet());
        this.clearPendingAssignmentsAndLocalNamesCache();
    }

    private void updateSubscription(SortedSet<TopicIdPartition> assignedPartitions) {
        SortedSet<TopicPartition> assignedTopicPartitions = this.toTopicPartitionSet(assignedPartitions);
        if (this.subscriptions.hasAutoAssignedPartitions()) {
            this.subscriptions.assignFromSubscribed(assignedTopicPartitions);
        }
        this.updateAssignment(assignedPartitions);
    }

    public void transitionToJoining() {
        if (this.state == MemberState.FATAL) {
            this.log.warn("No action taken to join the group with the updated subscription because the member is in FATAL state");
            return;
        }
        if (this.reconciliationInProgress) {
            this.rejoinedWhileReconciliationInProgress = true;
        }
        this.resetEpoch();
        this.transitionTo(MemberState.JOINING);
        this.clearPendingAssignmentsAndLocalNamesCache();
    }

    public CompletableFuture<Void> leaveGroup() {
        if (this.isNotInGroup()) {
            if (this.state == MemberState.FENCED) {
                this.clearSubscription();
                this.transitionTo(MemberState.UNSUBSCRIBED);
            }
            return CompletableFuture.completedFuture(null);
        }
        if (this.state == MemberState.PREPARE_LEAVING || this.state == MemberState.LEAVING) {
            return this.leaveGroupInProgress.get();
        }
        this.transitionTo(MemberState.PREPARE_LEAVING);
        CompletableFuture<Void> leaveResult = new CompletableFuture<Void>();
        this.leaveGroupInProgress = Optional.of(leaveResult);
        this.clearSubscription();
        this.transitionToSendingLeaveGroup(false);
        return leaveResult;
    }

    public void transitionToSendingLeaveGroup(boolean dueToExpiredPollTimer) {
        if (this.state == MemberState.FATAL) {
            this.log.warn("Member {} with epoch {} won't send leave group request because it is in FATAL state", (Object)this.memberId, (Object)this.memberEpoch);
            return;
        }
        if (this.state == MemberState.UNSUBSCRIBED) {
            this.log.warn("Member {} won't send leave group request because it is already out of the group.", (Object)this.memberId);
            return;
        }
        if (dueToExpiredPollTimer) {
            this.isPollTimerExpired = true;
            this.transitionTo(MemberState.PREPARE_LEAVING);
        }
        this.updateMemberEpoch(-1);
        this.currentAssignment = new HashMap<Uuid, SortedSet<Integer>>();
        this.transitionTo(MemberState.LEAVING);
    }

    private void notifyEpochChange(Optional<Integer> epoch, Optional<String> memberId) {
        this.stateUpdatesListeners.forEach(stateListener -> stateListener.onMemberEpochUpdated(epoch, memberId));
    }

    public boolean shouldHeartbeatNow() {
        MemberState state = this.state();
        return state == MemberState.ACKNOWLEDGING || state == MemberState.LEAVING || state == MemberState.JOINING;
    }

    public void onHeartbeatRequestSent() {
        MemberState state = this.state();
        if (state == MemberState.ACKNOWLEDGING) {
            if (this.targetAssignmentReconciled()) {
                this.transitionTo(MemberState.STABLE);
            } else {
                this.log.debug("Member {} with epoch {} transitioned to {} after a heartbeat was sent to ack a previous reconciliation. New assignments are ready to be reconciled.", new Object[]{this.memberId, this.memberEpoch, MemberState.RECONCILING});
                this.transitionTo(MemberState.RECONCILING);
            }
        } else if (state == MemberState.LEAVING) {
            if (this.isPollTimerExpired) {
                this.transitionToStale();
            } else {
                this.transitionToUnsubscribed();
            }
        }
    }

    public void onHeartbeatRequestSkipped() {
        if (this.state == MemberState.LEAVING) {
            this.log.debug("Heartbeat for leaving group could not be sent. Member {} with epoch {} will transition to {}.", new Object[]{this.memberId, this.memberEpoch, MemberState.UNSUBSCRIBED});
            this.transitionToUnsubscribed();
        }
    }

    private void transitionToUnsubscribed() {
        this.transitionTo(MemberState.UNSUBSCRIBED);
        this.leaveGroupInProgress.get().complete(null);
        this.leaveGroupInProgress = Optional.empty();
    }

    private boolean targetAssignmentReconciled() {
        return this.currentAssignment.equals(this.currentTargetAssignment);
    }

    public boolean shouldSkipHeartbeat() {
        MemberState state = this.state();
        return state == MemberState.UNSUBSCRIBED || state == MemberState.FATAL || state == MemberState.STALE || state == MemberState.FENCED;
    }

    public boolean isLeavingGroup() {
        MemberState state = this.state();
        return state == MemberState.PREPARE_LEAVING || state == MemberState.LEAVING;
    }

    public void maybeRejoinStaleMember() {
        this.isPollTimerExpired = false;
        if (this.state == MemberState.STALE) {
            this.log.debug("Expired poll timer has been reset so stale member {} will rejoin the groupwhen it completes releasing its previous assignment.", (Object)this.memberId);
            this.transitionToJoining();
        }
    }

    public void transitionToStale() {
        this.transitionTo(MemberState.STALE);
        this.clearSubscription();
        this.log.debug("Member {} sent leave group heartbeat and released its assignment. It will remain in {} state until the poll timer is reset, and it will then rejoin the group", (Object)this.memberId, (Object)MemberState.STALE);
    }

    void maybeReconcile() {
        if (this.targetAssignmentReconciled()) {
            this.log.trace("Ignoring reconciliation attempt. Target assignment is equal to the current assignment.");
            return;
        }
        if (this.reconciliationInProgress) {
            this.log.trace("Ignoring reconciliation attempt. Another reconciliation is already in progress. Assignment {} will be handled in the next reconciliation loop.", this.currentTargetAssignment);
            return;
        }
        SortedSet<TopicIdPartition> assignedTopicIdPartitions = this.findResolvableAssignmentAndTriggerMetadataUpdate();
        TreeSet<TopicPartition> ownedPartitions = new TreeSet<TopicPartition>(TOPIC_PARTITION_COMPARATOR);
        ownedPartitions.addAll(this.subscriptions.assignedPartitions());
        SortedSet<TopicPartition> assignedTopicPartitions = this.toTopicPartitionSet(assignedTopicIdPartitions);
        boolean sameAssignmentReceived = assignedTopicPartitions.equals(ownedPartitions);
        if (sameAssignmentReceived) {
            this.log.debug("Ignoring reconciliation attempt. Target assignment ready to reconcile {} is equal to the member current assignment {}.", assignedTopicPartitions, ownedPartitions);
            return;
        }
        this.markReconciliationInProgress();
        TreeSet<TopicPartition> addedPartitions = new TreeSet<TopicPartition>(TOPIC_PARTITION_COMPARATOR);
        addedPartitions.addAll(assignedTopicPartitions);
        addedPartitions.removeAll(ownedPartitions);
        TreeSet<TopicPartition> revokedPartitions = new TreeSet<TopicPartition>(TOPIC_PARTITION_COMPARATOR);
        revokedPartitions.addAll(ownedPartitions);
        revokedPartitions.removeAll(assignedTopicPartitions);
        this.log.info("Updating assignment with\n\tAssigned partitions:                       {}\n\tCurrent owned partitions:                  {}\n\tAdded partitions (assigned - owned):       {}\n\tRevoked partitions (owned - assigned):     {}\n", new Object[]{assignedTopicIdPartitions, ownedPartitions, addedPartitions, revokedPartitions});
        if (!this.maybeAbortReconciliation()) {
            this.revokeAndAssign(assignedTopicIdPartitions, revokedPartitions);
        }
    }

    private void revokeAndAssign(SortedSet<TopicIdPartition> assignedTopicIdPartitions, SortedSet<TopicPartition> revokedPartitions) {
        if (!revokedPartitions.isEmpty()) {
            this.revokePartitions(revokedPartitions);
        }
        if (!this.maybeAbortReconciliation()) {
            this.assignPartitions(assignedTopicIdPartitions);
        }
        this.transitionTo(MemberState.ACKNOWLEDGING);
        this.markReconciliationCompleted();
    }

    boolean maybeAbortReconciliation() {
        boolean shouldAbort;
        boolean bl = shouldAbort = this.state != MemberState.RECONCILING || this.rejoinedWhileReconciliationInProgress;
        if (shouldAbort) {
            String reason = this.rejoinedWhileReconciliationInProgress ? "the member has re-joined the group" : "the member already transitioned out of the reconciling state into " + (Object)((Object)this.state);
            this.log.info("Interrupting reconciliation that is not relevant anymore because {}.", (Object)reason);
            this.markReconciliationCompleted();
        }
        return shouldAbort;
    }

    void updateAssignment(Set<TopicIdPartition> assignedTopicIdPartitions) {
        this.currentAssignment.clear();
        assignedTopicIdPartitions.forEach(topicIdPartition -> {
            Uuid topicId = topicIdPartition.topicId();
            this.currentAssignment.computeIfAbsent(topicId, k -> new TreeSet()).add(topicIdPartition.partition());
        });
    }

    private SortedSet<TopicPartition> toTopicPartitionSet(SortedSet<TopicIdPartition> topicIdPartitions) {
        TreeSet<TopicPartition> result = new TreeSet<TopicPartition>(TOPIC_PARTITION_COMPARATOR);
        topicIdPartitions.forEach(topicIdPartition -> result.add(topicIdPartition.topicPartition()));
        return result;
    }

    void markReconciliationInProgress() {
        this.reconciliationInProgress = true;
        this.rejoinedWhileReconciliationInProgress = false;
    }

    void markReconciliationCompleted() {
        this.reconciliationInProgress = false;
        this.rejoinedWhileReconciliationInProgress = false;
    }

    private SortedSet<TopicIdPartition> findResolvableAssignmentAndTriggerMetadataUpdate() {
        TreeSet<TopicIdPartition> assignmentReadyToReconcile = new TreeSet<TopicIdPartition>(TOPIC_ID_PARTITION_COMPARATOR);
        HashMap<Uuid, SortedSet<Integer>> unresolved = new HashMap<Uuid, SortedSet<Integer>>(this.currentTargetAssignment);
        Iterator<Map.Entry<Uuid, SortedSet<Integer>>> it = unresolved.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<Uuid, SortedSet<Integer>> e = it.next();
            Uuid topicId = e.getKey();
            SortedSet<Integer> topicPartitions = e.getValue();
            Optional<String> nameFromMetadata = this.findTopicNameInGlobalOrLocalCache(topicId);
            nameFromMetadata.ifPresent(resolvedTopicName -> {
                topicPartitions.forEach(tp -> assignmentReadyToReconcile.add(new TopicIdPartition(topicId, (int)tp, (String)resolvedTopicName)));
                it.remove();
            });
        }
        if (!unresolved.isEmpty()) {
            this.log.debug("Topic Ids {} received in target assignment were not found in metadata and are not currently assigned. Requesting a metadata update now to resolve topic names.", unresolved.keySet());
            this.metadata.requestUpdate(true);
        }
        return assignmentReadyToReconcile;
    }

    private Optional<String> findTopicNameInGlobalOrLocalCache(Uuid topicId) {
        String nameFromMetadataCache = this.metadata.topicNames().getOrDefault(topicId, null);
        if (nameFromMetadataCache != null) {
            this.assignedTopicNamesCache.put(topicId, nameFromMetadataCache);
            return Optional.of(nameFromMetadataCache);
        }
        String nameFromSubscriptionCache = this.assignedTopicNamesCache.getOrDefault(topicId, null);
        return Optional.ofNullable(nameFromSubscriptionCache);
    }

    void revokePartitions(Set<TopicPartition> revokedPartitions) {
        this.log.info("Revoking previously assigned partitions {}", (Object)revokedPartitions.stream().map(TopicPartition::toString).collect(Collectors.joining(", ")));
        this.markPendingRevocationToPauseFetching(revokedPartitions);
        if (this.state == MemberState.FATAL) {
            String errorMsg = String.format("Member %s with epoch %s received a fatal error while waiting for a revocation commit to complete. Will abort revocation.", this.memberId, this.memberEpoch);
            this.log.debug(errorMsg);
        }
    }

    private void assignPartitions(SortedSet<TopicIdPartition> assignedPartitions) {
        this.updateSubscription(assignedPartitions);
        Set assignedTopics = assignedPartitions.stream().map(TopicIdPartition::topic).collect(Collectors.toSet());
        this.assignedTopicNamesCache.values().retainAll(assignedTopics);
    }

    private void markPendingRevocationToPauseFetching(Set<TopicPartition> partitionsToRevoke) {
        this.log.debug("Marking partitions pending for revocation: {}", partitionsToRevoke);
        this.subscriptions.markPendingRevocation(partitionsToRevoke);
    }

    private void clearPendingAssignmentsAndLocalNamesCache() {
        this.currentTargetAssignment.clear();
        this.assignedTopicNamesCache.clear();
    }

    private void resetEpoch() {
        this.updateMemberEpoch(0);
    }

    private void updateMemberEpoch(int newEpoch) {
        boolean newEpochReceived = this.memberEpoch != newEpoch;
        this.memberEpoch = newEpoch;
        if (newEpochReceived) {
            if (this.memberEpoch > 0) {
                this.notifyEpochChange(Optional.of(this.memberEpoch), Optional.ofNullable(this.memberId));
            } else {
                this.notifyEpochChange(Optional.empty(), Optional.empty());
            }
        }
    }

    public MemberState state() {
        return this.state;
    }

    public Map<Uuid, SortedSet<Integer>> currentAssignment() {
        return this.currentAssignment;
    }

    Set<Uuid> topicsAwaitingReconciliation() {
        return this.topicPartitionsAwaitingReconciliation().keySet();
    }

    Map<Uuid, SortedSet<Integer>> topicPartitionsAwaitingReconciliation() {
        HashMap topicPartitionMap = new HashMap();
        this.currentTargetAssignment.forEach((topicId, targetPartitions) -> {
            SortedSet<Integer> reconciledPartitions = this.currentAssignment.get(topicId);
            if (!targetPartitions.equals(reconciledPartitions)) {
                TreeSet missingPartitions = new TreeSet(targetPartitions);
                if (reconciledPartitions != null) {
                    missingPartitions.removeAll(reconciledPartitions);
                }
                topicPartitionMap.put(topicId, missingPartitions);
            }
        });
        return Collections.unmodifiableMap(topicPartitionMap);
    }

    boolean reconciliationInProgress() {
        return this.reconciliationInProgress;
    }

    public void registerStateListener(MemberStateListener listener) {
        if (listener == null) {
            throw new IllegalArgumentException("State updates listener cannot be null");
        }
        this.stateUpdatesListeners.add(listener);
    }

    @Override
    public NetworkClientDelegate.PollResult poll(long currentTimeMs) {
        if (this.state == MemberState.RECONCILING) {
            this.maybeReconcile();
        }
        return NetworkClientDelegate.PollResult.EMPTY;
    }
}

